using System;
using System.Collections;
using System.Data;
using System.Reflection;

using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.ExceptionManagement;

using CANNEDCOMMENT = gov.va.med.vbecs.Common.VbecsTables.CannedComment;

namespace gov.va.med.vbecs.BOL
{

		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2002</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		///Canned Comment business layer class
		///</summary>

		#endregion


	public class CannedComment:BaseBusinessObject, IRecordStatusCode
	{
		#region Variables

		/// <summary>
		/// The textual content of the canned comment
		/// </summary>
		private string _comments;
		/// <summary>
		/// Identifies a category type of canned comment
		/// </summary>
		private string _cannedCommentType;
		/// <summary>
		/// A number that facilitates presenting values in a specified order
		/// </summary>
		private short _sortIndex;

		//private int _cannedCommentID;

		private Guid _cannedCommentGUID;

		private Common.RecordStatusCode _recordStatusCode;

		#endregion

		#region Constructors

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2401"> 
		///		<ExpectedInput>Valid canned comment data row</ExpectedInput>
		///		<ExpectedOutput>CannedComment object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2866"> 
		///		<ExpectedInput>Invalid canned comment data row</ExpectedInput>
		///		<ExpectedOutput>Empty canned comment object created</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates CannedComment object and populates it with data from DataRow
		/// </summary>
		public CannedComment(DataRow dr)
		{
			this.LoadFromDataRow(dr);
		}

		
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2480"> 
		///		<ExpectedInput>CannedCommentGuid</ExpectedInput>
		///		<ExpectedOutput>Initialized canned comment object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2482"> 
		///		<ExpectedInput>Invalid CannedCommentGuid</ExpectedInput>
		///		<ExpectedOutput>Empty CannedCommentObject</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Creates CannedComment object and populates it with data
		/// </summary>
		/// <param name="cannedCommentGuid"></param>
		public CannedComment(Guid cannedCommentGuid)
		{
			DataTable dtComments = DAL.CannedComment.GetCannedComment(cannedCommentGuid);
			//
			if (dtComments.Rows.Count == 0)
			{
				this._cannedCommentGUID = cannedCommentGuid;
				this.IsNew = true;
			}
			else
			{
				this.LoadFromDataRow(dtComments.Rows[0]);
				this.IsNew = false;
			}
		}

		#endregion

		#region Properties

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="856"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="857"> 
		///		<ExpectedInput>Empty String</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="858"> 
		///		<ExpectedInput>Invalid comment string</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets textual content of a canned comment
		/// </summary>
		public override string Comments
		{
			get
			{
				return this._comments;
			}
			set
			{
				if (value.Length > 255)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
											MethodBase.GetCurrentMethod().Name, true), "255").ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}
				if (value != this._comments)
				{
					this._comments = value;
					this.IsDirty = true;
				}
				RuleBroken(MethodBase.GetCurrentMethod().Name, value.Trim().Length == 0);
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="859"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="860"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets type of a canned comment
		/// </summary>
		public string CannedCommentType
		{
			get
			{
				return this._cannedCommentType;
			}
			set
			{
				if (value != this._cannedCommentType)
				{
					this._cannedCommentType = value;
					this.IsDirty = true;
				}
				//No brokenrules coded
				//RuleBroken(MethodBase.GetCurrentMethod().Name, value.Trim().Length==0);
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>01/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="263"> 
		///		<ExpectedInput>Common.RecordStatusCode</ExpectedInput>
		///		<ExpectedOutput>Common.RecordStatusCode</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2484"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// The status of the record
		/// </summary>
		public Common.RecordStatusCode RecordStatusCode
		{
			get
			{
				return _recordStatusCode;
			}
			set
			{
				_recordStatusCode = value;
			}
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="861"> 
		///		<ExpectedInput>short</ExpectedInput>
		///		<ExpectedOutput>short</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="862"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets specified order of a canned comment within its category
		/// </summary>
		public short SortIndex
		{
			get
			{
				return this._sortIndex;
			}
			set
			{
				if (value != this._sortIndex)
				{
					this._sortIndex = value;
					this.IsDirty = true;
				}
				//No broken rules coded
				//RuleBroken(MethodBase.GetCurrentMethod().Name, value < 1);
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>01/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="262"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2486"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets or sets the unique id of a canned comment
		/// </summary>
		public Guid CannedCommentGuid
		{
			get
			{
				return this._cannedCommentGUID;
			}
			set
			{
				this._cannedCommentGUID = value;
			}
		}


		#endregion

		#region Methods



		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/6/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="866"> 
		///		<ExpectedInput>Empty DataRow</ExpectedInput>
		///		<ExpectedOutput>DataRow with Object's values</ExpectedOutput>
		///	</Case>
		///	
		///	<Case type="1" testid ="2160"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public function which returns data row containing current canned comment object properties
		/// </summary>
		/// <returns>Data row containing canned comment information</returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.CannedCommentCategoryCode))
			{
				dtRow[CANNEDCOMMENT.CannedCommentCategoryCode] = this.CannedCommentType;
			}

			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.CannedCommentGuid))
			{
				dtRow[CANNEDCOMMENT.CannedCommentGuid] = this.CannedCommentGuid;
			}

			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.CannedCommentText))
			{
				dtRow[CANNEDCOMMENT.CannedCommentText] = this.Comments;	
			}

			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.DivisionCode))
			{
				dtRow[CANNEDCOMMENT.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			}

			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.LastUpdateDate))
			{
				dtRow[CANNEDCOMMENT.LastUpdateDate] = this.LastUpdateDate;
			}

			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.LastUpdateUser))
			{
				dtRow[CANNEDCOMMENT.LastUpdateUser] = this.LastUpdateUser;	
			}

			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.RecordStatusCode))
			{
				dtRow[CANNEDCOMMENT.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(this.RecordStatusCode);
			}

			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.RowVersion))
			{
				if (this.RowVersion != null)
				{
					dtRow[CANNEDCOMMENT.RowVersion] = this.RowVersion;
				}			
			}
			
			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.SortColumn))
			{
				dtRow[CANNEDCOMMENT.SortColumn] = this.SortIndex;
			}
		
			return dtRow;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2002</CreationDate>
		/// <summary>
		/// Method to set properties of a canned comment object based on supplied data row
		/// </summary>
		/// <param name="dtRow">Data row defining canned comment</param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.CannedCommentCategoryCode))
			{
				if (!dtRow.IsNull(CANNEDCOMMENT.CannedCommentCategoryCode))
				{
					this.CannedCommentType = dtRow[CANNEDCOMMENT.CannedCommentCategoryCode].ToString();
				}
			}

			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.CannedCommentGuid))
			{
				if (!dtRow.IsNull(CANNEDCOMMENT.CannedCommentGuid))
				{
					this.CannedCommentGuid = (System.Guid)dtRow[CANNEDCOMMENT.CannedCommentGuid];
				}
				else
				{
					this.CannedCommentGuid = System.Guid.Empty;
				}
			}

			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.CannedCommentText))
			{
				if (!dtRow.IsNull(CANNEDCOMMENT.CannedCommentText))
				{
					this.Comments = dtRow[CANNEDCOMMENT.CannedCommentText].ToString();
				}
			}

			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.LastUpdateDate))
			{
				if (!dtRow.IsNull(CANNEDCOMMENT.LastUpdateDate))
				{
					this.LastUpdateDate = (System.DateTime)dtRow[CANNEDCOMMENT.LastUpdateDate];
				}
			}

			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.LastUpdateUser))
			{
				if (!dtRow.IsNull(CANNEDCOMMENT.LastUpdateUser))
				{
					this.LastUpdateUser = dtRow[CANNEDCOMMENT.LastUpdateUser].ToString();
				}
			}

			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.RecordStatusCode))
			{
				if (!dtRow.IsNull(CANNEDCOMMENT.RecordStatusCode))
				{
					this.RecordStatusCode = Common.Utility.GetRecordStatusCodeFromString(dtRow[CANNEDCOMMENT.RecordStatusCode].ToString());
				}
			}

			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.RowVersion))
			{
				if (!dtRow.IsNull(CANNEDCOMMENT.RowVersion))
				{
					this.RowVersion = (byte[])dtRow[CANNEDCOMMENT.RowVersion];
				}
			}

			if (dtRow.Table.Columns.Contains(CANNEDCOMMENT.SortColumn))
			{
				if (!dtRow.IsNull(CANNEDCOMMENT.SortColumn))
				{
					this.SortIndex = (short)dtRow[CANNEDCOMMENT.SortColumn];
				}
				else
				{
					this.SortIndex = 0;
				}
			}

			this.IsDirty = false;
			this.IsNew = false;
		}

		
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="864"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Data table of canned comments for division</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2487"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public function which calls data access layer to retrieve all canned comments within a division
		/// Implements BR_31.01
		/// </summary>
		/// <returns>Data table containing canned comments</returns>
		public static DataTable GetCannedComments()
		{
			//BR_31.01
			DataTable dt = DAL.CannedComment.GetCannedComments(Common.LogonUser.LogonUserDivisionCode);//, returnCannedCommentID);

			dt.AcceptChanges();
			
			return dt;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="865"> 
		///		<ExpectedInput>Canned comment category string</ExpectedInput>
		///		<ExpectedOutput>Data table of canned comments for division and category</ExpectedOutput>
		///	</Case>
		///	
		///	<Case type="1" testid ="2159"> 
		///		<ExpectedInput>Invalid Canned comment category string</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public function which calls data access layer to retrieve canned comments of a specific category within a division
		/// Implements BR_31.01
		/// </summary>
		/// <param name="category">Canned Comment Category Type</param>
		/// <returns>Data table containing canned comments</returns>
		public static DataTable GetCannedComments(string category)
		{
			//BR_31.01
			return DAL.CannedComment.GetCannedComments(Common.LogonUser.LogonUserDivisionCode, category);
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2488"> 
		///		<ExpectedInput>Divison, Canned comment category string</ExpectedInput>
		///		<ExpectedOutput>Data table of canned comments for division and category</ExpectedOutput>
		///	</Case>
		///	
		///	<Case type="1" testid ="2489"> 
		///		<ExpectedInput>Invalid Division, Canned comment category string</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets a data table of canned comments given division code and category
		/// Implements BR_31.01
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="category"></param>
		/// <returns></returns>
		public static DataTable GetCannedComments(string divisionCode, string category)
		{
			//BR_31.01
			return DAL.CannedComment.GetCannedComments(divisionCode, category);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/31/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4056"> 
		///		<ExpectedInput>Divison, Canned comment category string</ExpectedInput>
		///		<ExpectedOutput>ArrayLIst of hashtables</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2490"> 
		///		<ExpectedInput>Invalid Divison, Canned comment category string</ExpectedInput>
		///		<ExpectedOutput>Empty Arraylist</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets Canned Comments in ArrayList of Hashtables
		/// To be used for read-only access of Canned Comment Info
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="category"></param>
		/// <returns>ArrayList containing canned comments</returns>
		public static ArrayList GetCannedCommentsList(string divisionCode, string category)
		{
			DataTable dt = DAL.CannedComment.GetCannedComments(divisionCode, category);
			
			ArrayList comments = new ArrayList(dt.Rows.Count);

			foreach(DataRow dr in dt.Rows)
			{
				Hashtable record = new Hashtable();
				record.Add(Common.VbecsTables.CannedComment.CannedCommentGuid, dr[Common.VbecsTables.CannedComment.CannedCommentGuid]);
				record.Add(Common.VbecsTables.CannedComment.CannedCommentText, dr[Common.VbecsTables.CannedComment.CannedCommentText]);
				record.Add(Common.VbecsTables.CannedComment.CannedCommentCategoryCode, dr[Common.VbecsTables.CannedComment.CannedCommentCategoryCode]);
				record.Add(Common.VbecsTables.CannedComment.SortColumn, dr[Common.VbecsTables.CannedComment.SortColumn]);
				record.Add(Common.VbecsTables.CannedComment.DivisionCode, dr[Common.VbecsTables.CannedComment.DivisionCode]);
				record.Add(Common.VbecsTables.CannedComment.RecordStatusCode, dr[Common.VbecsTables.CannedComment.RecordStatusCode]);
				record.Add(Common.VbecsTables.CannedComment.RowVersion, dr[Common.VbecsTables.CannedComment.RowVersion]);
				comments.Add(record);
			}

			return comments;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/31/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2496"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2497"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public statuc function which calls data access layer to retrieve all canned comment category types
		/// </summary>
		/// <returns>Data table containing canned comment categories</returns>
		public static DataTable GetCannedCommentCategories()
		{
			return DAL.CannedComment.GetCannedCommentCategories();
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/6/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="867"> 
		///		<ExpectedInput>Data table, UpdateFunctionId</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2166"> 
		///		<ExpectedInput>Invalid RowVersion</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public function which calls data access layer to save changes to a table of canned comments to file and return success indicator
		/// </summary>
		/// <param name="cannedComments">Data table containing canned comments</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns>Success indicator</returns>
		public static bool Save(DataTable cannedComments, Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Collections.ArrayList alData = new System.Collections.ArrayList();
			System.Collections.ArrayList alSproc = new System.Collections.ArrayList();

			DataTable modifiedComments = Common.Utility.AppendLastUpdateInformation(cannedComments.GetChanges(DataRowState.Modified), lastUpdateFunctionId);
			if (modifiedComments != null)
			{
				System.Data.DataTable dtUpdateComments = DAL.CannedComment.GetEmptyCannedCommentTableSchema(true);

				foreach(DataRow dr in modifiedComments.Rows)
				{
					System.Data.DataRow drUpd = dtUpdateComments.NewRow();

					drUpd[CANNEDCOMMENT.CannedCommentGuid] = dr[CANNEDCOMMENT.CannedCommentGuid];
					drUpd[CANNEDCOMMENT.SortColumn] = dr[CANNEDCOMMENT.SortColumn];
					drUpd[CANNEDCOMMENT.RecordStatusCode] = dr[CANNEDCOMMENT.RecordStatusCode];
					drUpd[CANNEDCOMMENT.RowVersion] = dr[CANNEDCOMMENT.RowVersion];
					drUpd[CANNEDCOMMENT.LastUpdateFunctionId] = lastUpdateFunctionId;
					
					dtUpdateComments.Rows.Add(drUpd);
				}
				alData.Add(dtUpdateComments);
				alSproc.Add(Common.VbecsStoredProcs.UpdateCannedComment.StoredProcName);
			}
			//
			DataTable addedComments = Common.Utility.AppendLastUpdateInformation(cannedComments.GetChanges(DataRowState.Added), lastUpdateFunctionId);
			if (addedComments != null)
			{
				System.Data.DataTable dtInsertComments = DAL.CannedComment.GetEmptyCannedCommentTableSchema(false);
				
				foreach(DataRow dr in addedComments.Rows)
				{
					System.Data.DataRow drIns = dtInsertComments.NewRow();

					drIns[CANNEDCOMMENT.CannedCommentGuid] = dr[CANNEDCOMMENT.CannedCommentGuid];
					drIns[CANNEDCOMMENT.CannedCommentCategoryCode] = dr[CANNEDCOMMENT.CannedCommentCategoryCode];
					drIns[CANNEDCOMMENT.CannedCommentText] = dr[CANNEDCOMMENT.CannedCommentText];
					drIns[CANNEDCOMMENT.SortColumn] = dr[CANNEDCOMMENT.SortColumn];
					drIns[CANNEDCOMMENT.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drIns[CANNEDCOMMENT.RecordStatusCode] = dr[CANNEDCOMMENT.RecordStatusCode];
					drIns[CANNEDCOMMENT.LastUpdateFunctionId] = lastUpdateFunctionId;
					
					dtInsertComments.Rows.Add(drIns);
				}

				alData.Add(dtInsertComments);
				alSproc.Add(Common.VbecsStoredProcs.InsertCannedComment.StoredProcName);
			}
			return DAL.CannedComment.Save(alSproc, alData);
		}

		#endregion

		
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/6/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2503"> 
		///		<ExpectedInput>Valid canned comment Guid, valid free text string</ExpectedInput>
		///		<ExpectedOutput>Combined canned and free-text comment</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2504"> 
		///		<ExpectedInput>Invalid canned comment guid, no free-text string</ExpectedInput>
		///		<ExpectedOutput>Empty string</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Static function combines a canned comment and a free text comment to a single string
		/// </summary>
		/// <param name="cannedCommentGuid"></param>
		/// <param name="freeTextComment"></param>
		/// <returns></returns>
		public static string CombinedComment(Guid cannedCommentGuid, string freeTextComment)
		{
			BOL.CannedComment cannedComment = new BOL.CannedComment(cannedCommentGuid);

			if (cannedComment.Comments != null)
			{
				if ((cannedComment.Comments.Trim().Length == 0) || (cannedComment.Comments.Trim().ToUpper() == "OTHER"))
				{
					return freeTextComment;
				}
				if (freeTextComment.Trim().Length == 0)
				{
					return cannedComment.Comments;
				}
				if (!cannedComment.Comments.EndsWith("."))
				{
					cannedComment.Comments += ".";
				}
				return cannedComment.Comments + " " + freeTextComment;
			}
			else
			{
				return freeTextComment;
			}
		}
	}
}
